<?php

namespace App\Livewire;

use App\Models\InventoryConsumptionRequest;
use App\Models\InventoryItem;
use App\Models\LabNotebookEntry;
use App\Models\Project;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Livewire\Component;
use Livewire\WithPagination;
use App\Policies\WarehousePolicy;
use Illuminate\Support\Collection;


class WarehouseRequestManager extends Component
{
    use AuthorizesRequests;
    use WithPagination;

    public ?InventoryConsumptionRequest $requestToProcess = null;
    public $confirmingApproval = false;
    public $confirmingRejection = false;
    public $rejectionNotes = '';


    // [NEW] Properties for the new review modal
    public ?LabNotebookEntry $reviewingLabNote = null;
    public Collection $requestsForNote;
    public bool $confirmingNoteReview = false;
    // [NEW] Properties for inline editing
    public $editingRequestId = null;
    public $newQuantity = '';
    // [NEW] Properties for Budget Override Modal
    public bool $confirmingBudgetOverride = false;
    public string $overrideWarningMessage = '';
    protected $paginationTheme = 'bootstrap';

    public function confirmApproval(InventoryConsumptionRequest $request)
    {
        $this->authorize('process', $request);
        $this->requestToProcess = $request;

        $inventoryItem = $request->inventoryItem;
        $project = $request->project;
        $user = Auth::user();

        // بررسی موجودی انبار
        if ($request->type === 'consumption' && $inventoryItem->quantity < $request->quantity) {
            session()->flash('error', "موجودی انبار برای '{$inventoryItem->name}' کافی نیست.");
            return;
        }

        // بررسی سقف بودجه
        $budgetItem = $project->budgetItems()->where('inventory_item_id', $inventoryItem->id)->first();
        if ($budgetItem) {
            $remainingBudget = $budgetItem->pivot->max_allowed_quantity - $budgetItem->pivot->total_consumed_quantity;
            if ($request->type === 'consumption' && $request->quantity > $remainingBudget) {
                // اگر کاربر مالک طرح است، مودال هشدار را نمایش بده
                if ($user->id === $project->owner_id) {
                    $this->overrideWarningMessage = "مقدار درخواستی از سقف بودجه بیشتر است. با تایید، سقف بودجه به صورت خودکار افزایش خواهد یافت. آیا مطمئن هستید؟";
                    $this->confirmingBudgetOverride = true;
                } else {
                    // در غیر این صورت، به کاربر عادی خطا نمایش بده
                    session()->flash('error', "مقدار درخواستی برای '{$inventoryItem->name}' از سقف مجاز بودجه فراتر می‌رود. لطفاً با مالک طرح تماس بگیرید.");
                }
                return;
            }
        }

        // اگر هیچ مشکلی نبود، مودال تایید عادی را نمایش بده
        $this->confirmingApproval = true;
    }

    public function approveRequest()
    {
        if (!$this->requestToProcess) return;
        $this->authorize('process', $this->requestToProcess);


        $request = $this->requestToProcess;
        $inventoryItem = $request->inventoryItem;

        // برای درخواست مصرف، ابتدا موجودی انبار را چک می‌کنیم
        if ($request->type === 'consumption' && $inventoryItem->quantity < $request->quantity) {
            session()->flash('error', "موجودی انبار برای '{$inventoryItem->name}' کافی نیست.");
            $this->confirmingApproval = false;
            return;
        }

        DB::transaction(function () use ($request, $inventoryItem) {
            if ($request->type === 'consumption') {
                // کسر از موجودی انبار
                $inventoryItem->decrement('quantity', $request->quantity);
                // افزایش مقدار مصرف شده در بودجه طرح
                $request->project->budgetItems()->where('inventory_item_id', $inventoryItem->id)->increment('total_consumed_quantity', $request->quantity);
            } else { // type === 'return'
                // اضافه به موجودی انبار
                $inventoryItem->increment('quantity', $request->quantity);
                // کسر از مقدار مصرف شده در بودجه طرح
                $request->project->budgetItems()->where('inventory_item_id', $inventoryItem->id)->decrement('total_consumed_quantity', $request->quantity);
            }

            // به‌روزرسانی وضعیت درخواست
            $request->update([
                'status' => 'approved',
                'approver_id' => Auth::id(),
                'processed_at' => now(),
            ]);
        });

        $this->confirmingApproval = false;
        session()->flash('message', 'درخواست با موفقیت تایید شد.');
        // Refresh the modal data
        $this->reviewNoteRequests($this->requestToProcess->labNotebookEntry->id);
    }

    public function confirmRejection($requestId)
    {
        $request = InventoryConsumptionRequest::findOrFail($requestId);
        $this->authorize('process', $request);
        $this->resetErrorBag();
        $this->requestToProcess = $request;
        $this->rejectionNotes = '';
        $this->confirmingRejection = true;
    }

    public function rejectRequest()
    {
        if (!$this->requestToProcess) return;
        $this->authorize('process', $this->requestToProcess);

        $this->validate(['rejectionNotes' => 'required|string|min:10']);

        DB::transaction(function () {
            // 1. Update the request status to rejected
            $this->requestToProcess->update([
                'status' => 'rejected',
                'approver_id' => Auth::id(),
                'processed_at' => now(),
                'notes' => $this->rejectionNotes,
            ]);

            $labNote = $this->requestToProcess->labNotebookEntry;
            $itemId = $this->requestToProcess->inventory_item_id;
            $rejectedQuantity = $this->requestToProcess->quantity;

            // 2. Find the current pivot record for the material in the lab note
            $pivotRecord = $labNote->materials()->where('inventory_item_id', $itemId)->first();

            if ($pivotRecord) {
                $newQuantityUsed = $pivotRecord->pivot->quantity_used - $rejectedQuantity;

                // 3. If the new quantity is positive, update the pivot table.
                //    Otherwise, detach the material completely.
                if ($newQuantityUsed > 0) {
                    $labNote->materials()->updateExistingPivot($itemId, ['quantity_used' => $newQuantityUsed]);
                } else {
                    $labNote->materials()->detach($itemId);
                }
            }
        });

        $this->confirmingRejection = false;
        session()->flash('message', 'درخواست رد شد و مقدار مصرفی در یادداشت اصلاح گردید.');
        $this->reviewNoteRequests($this->requestToProcess->labNotebookEntry->id);
    }


    public function reviewNoteRequests($labNoteId)
    {
        $labNote = LabNotebookEntry::findOrFail($labNoteId);
        $this->reviewingLabNote = $labNote->load('project');

        // [FIX] استفاده از رابطه جدید برای واکشی تمام درخواست‌ها
        $this->requestsForNote = $labNote->consumptionRequests()
            ->with(['requester', 'inventoryItem'])
            ->where('status', 'pending')
            ->get();
        $this->confirmingNoteReview = true;
    }

    /**
     * [NEW] Enable inline editing for a specific request.
     */
    public function startEditing($requestId)
    {
        $request = InventoryConsumptionRequest::findOrFail($requestId);
        $this->editingRequestId = $requestId;
        $this->newQuantity = $request->quantity;
    }

    /**
     * [NEW] Cancel inline editing.
     */
    public function cancelEditing()
    {
        $this->editingRequestId = null;
        $this->newQuantity = '';
        $this->resetErrorBag();
    }

    /**
     * [NEW] Update a request with a new quantity and handle conditional logic.
     */
    public function updateAndApproveRequest()
    {
        if (!$this->editingRequestId) return;
        $request = InventoryConsumptionRequest::findOrFail($this->editingRequestId);
        $this->authorize('process', $request);

        $this->validate(['newQuantity' => 'required|numeric|min:0']);

        $originalQuantity = $request->quantity;
        $newQuantity = (float) $this->newQuantity;

        // سناریو ۱: مقدار جدید کمتر یا مساوی مقدار درخواستی است
        if ($newQuantity <= $originalQuantity) {
            DB::transaction(function () use ($request, $newQuantity, $originalQuantity) {
                $inventoryItem = $request->inventoryItem;
                $labNote = $request->labNotebookEntry;
                $difference = $originalQuantity - $newQuantity;

                // تایید مستقیم مقدار جدید
                if ($request->type === 'consumption') {
                    if ($inventoryItem->quantity < $newQuantity) {
                        $this->addError('newQuantity', 'موجودی انبار کافی نیست.');
                        return;
                    }
                    $inventoryItem->decrement('quantity', $newQuantity);
                    $request->project->budgetItems()->where('inventory_item_id', $inventoryItem->id)->increment('total_consumed_quantity', $newQuantity);
                } else { // Return
                    $inventoryItem->increment('quantity', $newQuantity);
                    $request->project->budgetItems()->where('inventory_item_id', $inventoryItem->id)->decrement('total_consumed_quantity', $newQuantity);
                }

                // اصلاح مقدار مصرفی در یادداشت
                $pivotRecord = $labNote->materials()->where('inventory_item_id', $request->inventory_item_id)->first();
                if ($pivotRecord) {
                    $finalNoteQuantity = $pivotRecord->pivot->quantity_used - $difference;
                    $labNote->materials()->updateExistingPivot($request->inventory_item_id, ['quantity_used' => $finalNoteQuantity]);
                }

                // به‌روزرسانی درخواست اصلی با مقدار جدید
                $request->update(['quantity' => $newQuantity, 'status' => 'approved', 'approver_id' => Auth::id(), 'processed_at' => now(), 'notes' => 'مقدار توسط مسئول انبار اصلاح و تایید شد.']);
            });
        }
        // سناریو ۲: مقدار جدید بیشتر از مقدار درخواستی است
        else {
            $difference = $newQuantity - $originalQuantity;
            DB::transaction(function () use ($request, $originalQuantity, $difference, $newQuantity) {
                // ۱. تایید مقدار اولیه درخواست
                $inventoryItem = $request->inventoryItem;
                if ($inventoryItem->quantity < $originalQuantity) {
                    $this->addError('newQuantity', 'موجودی انبار برای مقدار اولیه کافی نیست.');
                    return;
                }
                $inventoryItem->decrement('quantity', $originalQuantity);
                $request->project->budgetItems()->where('inventory_item_id', $inventoryItem->id)->increment('total_consumed_quantity', $originalQuantity);
                $request->update(['status' => 'approved', 'approver_id' => Auth::id(), 'processed_at' => now(), 'notes' => 'مقدار اولیه تایید شد. درخواست افزایش ایجاد گردید.']);

                // ۲. ایجاد درخواست جدید برای مقدار مازاد جهت تایید مالک
                InventoryConsumptionRequest::create(['project_id' => $request->project_id, 'lab_notebook_entry_id' => $request->lab_notebook_entry_id, 'requester_id' => Auth::id(), 'inventory_item_id' => $request->inventory_item_id, 'type' => 'consumption', 'quantity' => $difference, 'status' => 'pending']);

                // ۳. به‌روزرسانی مقدار نهایی در یادداشت
                $request->labNotebookEntry->materials()->updateExistingPivot($request->inventory_item_id, ['quantity_used' => $newQuantity]);
            });
        }

        if ($this->getErrorBag()->isEmpty()) {
            $this->cancelEditing();
            $this->reviewNoteRequests($request->labNotebookEntry->id);
            session()->flash('message', 'عملیات با موفقیت انجام شد.');
        }
    }

    /**
     * [NEW] This method approves the request AND overrides the budget. Only for the project owner.
     */
    public function approveAndOverrideBudget()
    {
        if (!$this->requestToProcess) return;
        $this->authorize('process', $this->requestToProcess);

        $request = $this->requestToProcess;
        $inventoryItem = $request->inventoryItem;
        $project = $request->project;

        // [NEW] بررسی موجودی انبار قبل از افزایش بودجه
        if ($inventoryItem->quantity < $request->quantity) {
            session()->flash('error', "امکان افزایش بودجه وجود ندارد. موجودی انبار برای '{$inventoryItem->name}' کافی نیست.");
            $this->confirmingBudgetOverride = false;
            return;
        }

        DB::transaction(function () use ($request, $inventoryItem, $project) {
            $budgetItem = $project->budgetItems()->where('inventory_item_id', $inventoryItem->id)->first()->pivot;
            $neededIncrease = $request->quantity - ($budgetItem->max_allowed_quantity - $budgetItem->total_consumed_quantity);

            // 1. افزایش سقف بودجه
            $project->budgetItems()->updateExistingPivot($inventoryItem->id, [
                'max_allowed_quantity' => $budgetItem->max_allowed_quantity + $neededIncrease
            ]);

            // 2. کسر از موجودی انبار
            $inventoryItem->decrement('quantity', $request->quantity);

            // 3. افزایش مقدار مصرف شده در بودجه
            $project->budgetItems()->where('inventory_item_id', $inventoryItem->id)->increment('total_consumed_quantity', $request->quantity);

            // 4. به‌روزرسانی وضعیت درخواست
            $request->update(['status' => 'approved', 'approver_id' => Auth::id(), 'processed_at' => now(), 'notes' => 'با مجوز مالک، سقف بودجه افزایش یافت.']);
        });

        $this->confirmingBudgetOverride = false;
        session()->flash('message', 'درخواست با موفقیت تایید و سقف بودجه افزایش یافت.');

        // [FIX] Pass the ID, not the model instance, to prevent 404 errors.
        $this->reviewNoteRequests($this->requestToProcess->labNotebookEntry->id);
    }

    public function render()
    {
        $user = Auth::user();

        // 1. دریافت ID طرح‌هایی که کاربر مالک آن‌هاست یا مسئول انبار آن‌ها
        $managedProjectIds = Project::where('owner_id', $user->id)
            ->orWhere('warehouse_manager_id', $user->id)
            ->pluck('id');

        // 2. دریافت ID طرح‌هایی که کاربر در آن‌ها دسترسی مستقیم مدیریت انبار را دارد
        $projectIdsWithDirectPermission = $user->projectPermissions()
            ->where('name', 'warehouse-manage-requests')
            ->get()
            ->pluck('pivot.project_id');

        // 3. ادغام لیست IDها و حذف موارد تکراری
        $allManagedProjectIds = $managedProjectIds->merge($projectIdsWithDirectPermission)->unique();

        $labNoteIdsWithPendingRequests = InventoryConsumptionRequest::where('status', 'pending')
            ->whereIn('project_id', $allManagedProjectIds)
            ->distinct()
            ->pluck('lab_notebook_entry_id');


        $labNotesWithPendingRequests = LabNotebookEntry::with(['project', 'user', 'consumptionRequests' => function ($query) {
            // شمارش تعداد درخواست‌های فعال برای نمایش در view
            $query->where('status', 'pending');
        }])
            ->whereIn('id', $labNoteIdsWithPendingRequests)
            ->latest()
            ->paginate(5, ['*'], 'pendingPage');

        // Fetch processed requests for the history table
        $processedRequests = InventoryConsumptionRequest::with(['project', 'requester', 'inventoryItem', 'labNotebookEntry', 'approver'])
            ->where('status', '!=', 'pending')
            ->whereIn('project_id', $allManagedProjectIds)
            ->latest('processed_at')
            ->paginate(10, ['*'], 'processedPage');

        return view('livewire.warehouse-request-manager', [
            'labNotesWithPendingRequests' => $labNotesWithPendingRequests,
            'processedRequests' => $processedRequests,
        ])->layout('layouts.app');
    }
}
