<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Equipment extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'research_team_id',
        'name',
        'model',
        'serial_number',
        'system_sku',
        'user_defined_id',
        'responsible_user_id',
        'manager_id',
        'last_modified_by',
        'calibration_interval_days',
        'last_calibration_date',
        'next_calibration_date',
        // New fields for temp log
        'storage_location_id',
        'requires_temp_log',
        'min_temp',
        'max_temp',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'last_calibration_date' => 'date',
        'next_calibration_date' => 'date',
        'requires_temp_log' => 'boolean',
    ];

    /**
     * Get the research team that the equipment belongs to.
     */
    public function researchTeam(): BelongsTo
    {
        return $this->belongsTo(ResearchTeam::class);
    }

    /**
     * Get the user who is responsible for the equipment.
     */
    public function responsibleUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'responsible_user_id');
    }
    /**
     * [NEW] Get the manager who owns/registered the equipment.
     */
    public function manager(): BelongsTo
    {
        return $this->belongsTo(User::class, 'manager_id');
    }


    /**
     * Get the calibration logs for the equipment.
     */
    public function calibrationLogs(): HasMany
    {
        return $this->hasMany(CalibrationLog::class)->latest('calibration_date');
    }
    /**
     * [NEW] Get the user who last modified the equipment record.
     */
    public function lastModifiedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'last_modified_by');
    }
    /**
     * [NEW] Get the storage location (room) where the equipment is located.
     * This is the missing relationship that caused the error.
     */
    public function location(): BelongsTo
    {
        return $this->belongsTo(StorageLocation::class, 'storage_location_id');
    }
    /**
     * [NEW] Get the temperature logs for the equipment.
     * This is the missing relationship that caused the error.
     */
    public function temperatureLogs(): HasMany
    {
        return $this->hasMany(TemperatureLog::class);
    }
}
