<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;

class InventoryItem extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'sku',
        'item_type',
        'is_biological',
        'quantity',
        'unit',
        'price',
        'brand',
        'lot_number',
        'passage_number',
        'freeze_date',
        'cell_type',
        'media_used',
        'location_id',
        'research_team_id',
        'position_in_location',
        'storage_location_id',
        'owner_id',
        'received_date',
        'expiration_date',
        'minimum_stock_level',
        'predicted_stockout_date',
    ];
    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        // ... all your other casts
        'predicted_stockout_date' => 'date', // [NEW] Cast the new field to a date object
    ];

    /**
     * Get the location that the item belongs to.
     */
    public function location(): BelongsTo
    {
        return $this->belongsTo(StorageLocation::class, 'location_id');
    }

    public function researchTeam(): BelongsTo
    {
        return $this->belongsTo(ResearchTeam::class);
    }


    /**
     * The lab notebook entries that this item was used in.
     */
    public function notebookEntries(): BelongsToMany
    {
        return $this->belongsToMany(LabNotebookEntry::class, 'entry_material')
            ->withPivot('quantity_used')
            ->withTimestamps();
    }
    /**
     * Get the transfer items associated with the inventory item.
     */
    public function transferItems(): HasMany
    {
        return $this->hasMany(InventoryTransferItem::class);
    }
    public function acknowledgements(): HasMany
    {
        return $this->hasMany(LowStockAcknowledgement::class);
    }
    /**
     * تعریف رابطه چند به چند با طرح‌ها.
     * نشان می‌دهد که این آیتم انبار در بودجه کدام طرح‌ها قرار دارد.
     */
    public function projects()
    {
        return $this->belongsToMany(Project::class, 'project_inventory_item')
            ->withPivot('max_allowed_quantity', 'total_consumed_quantity')
            ->withTimestamps()
            ->using(ProjectInventoryItem::class);
    }

    /**
     * [NEW] Get the consumption requests for the inventory item.
     * This is the missing relationship that caused the error.
     */
    public function consumptionRequests(): HasMany
    {
        return $this->hasMany(InventoryConsumptionRequest::class);
    }
}
