<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Spatie\Tags\HasTags;

class LabNotebookEntry extends Model
{
    use HasFactory;
    use HasTags;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'sku',
        'title',
        'entry_date',
        'body',
        'user_id',
        'is_locked',
        'locked_at',
        'locked_by',
    ];
    /**
     * [NEW] Processes the body content to convert [[SKU]] syntax into clickable links.
     * This accessor is automatically called whenever you access $entry->processed_body.
     *
     * @return string The processed HTML content.
     */
    public function getProcessedBodyAttribute(): string
    {
        if (!$this->body) {
            return '';
        }

        // Use a static cache to avoid redundant database queries for the same SKU within a single request.
        static $entryCache = [];

        return preg_replace_callback(
            '/\[\[(.*?)\]\]/', // Matches any text inside [[...]]
            function ($matches) use (&$entryCache) {
                $sku = trim($matches[1]);
                if (empty($sku)) {
                    return $matches[0]; // Return original if empty, e.g., [[]]
                }

                // Check cache first to improve performance
                if (!isset($entryCache[$sku])) {
                    // Find the linked entry but exclude the current entry itself
                    $entryCache[$sku] = LabNotebookEntry::where('sku', $sku)
                        ->where('id', '!=', $this->id)
                        ->first();
                }

                $linkedEntry = $entryCache[$sku];

                if ($linkedEntry) {
                    // If found, create a link that triggers a Livewire action to show a preview modal.
                    return sprintf(
                        '<a href="#" wire:click.prevent="showEntryPreview(%d)" class="text-blue-600 dark:text-blue-400 font-semibold hover:underline" title="مشاهده پیش‌نمایش: %s">[[%s]]</a>',
                        $linkedEntry->id,
                        htmlspecialchars($linkedEntry->title), // Tooltip with the entry title
                        htmlspecialchars($sku)
                    );
                }

                // If not found, return a styled span indicating a broken link.
                return sprintf(
                    '<span class="text-red-500 dark:text-red-400 line-through" title="یادداشت با شناسه %s یافت نشد">[[%s]]</span>',
                    htmlspecialchars($sku),
                    htmlspecialchars($sku)
                );
            },
            $this->body
        );
    }

    /**
     * Get the route key for the model.
     * این متد به لاراول می‌گوید که از ستون 'sku' برای پیدا کردن مدل در URL استفاده کند.
     *
     * @return string
     */
    public function getRouteKeyName()
    {
        return 'sku';
    }
    /**
     * The materials and reagents used in this entry.
     */
    public function materials(): BelongsToMany
    {
        return $this->belongsToMany(InventoryItem::class, 'entry_material')
            ->withPivot('quantity_used')
            ->withTimestamps();
    }

    /**
     * Get the user that owns the lab notebook entry.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the project that the lab notebook entry belongs to.
     * این متد جدید، رابطه "تعلق داشتن به" را با مدل Project تعریف می‌کند.
     */
    public function project(): BelongsTo
    {
        return $this->belongsTo(Project::class);
    }

    /**
     * Get all of the versions for the lab notebook entry.
     */
    public function versions(): HasMany
    {
        return $this->hasMany(LabNotebookEntryVersion::class);
    }

    /**
     * Get the signatures for the lab notebook entry.
     */
    public function signatures(): HasMany
    {
        return $this->hasMany(LabNotebookSignature::class);
    }

    // در مدل App\Models\LabNotebookEntry
    public function consumptionRequests()
    {
        return $this->hasMany(InventoryConsumptionRequest::class, 'lab_notebook_entry_id');
    }
    /**
     * [NEW] The tasks that are associated with this lab notebook entry.
     * وظایفی که به این یادداشت آزمایشگاهی مرتبط شده‌اند.
     */
    public function tasks(): BelongsToMany
    {
        return $this->belongsToMany(Task::class, 'lab_notebook_entry_task', 'lab_notebook_entry_id', 'task_id');
    }

    /**
     * [NEW] Get the attachments for the lab notebook entry.
     */
    public function attachments(): HasMany
    {
        return $this->hasMany(Attachment::class, 'lab_notebook_entry_id');
    }

    public function comments(): HasMany
    {
        // This defines the one-to-many relationship with comments.
        return $this->hasMany(LabNotebookComment::class);
    }

}
