<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany; // این خط اضافه می‌شود
use Illuminate\Database\Eloquent\Relations\MorphToMany;
use Illuminate\Database\Eloquent\Relations\HasOne;

class Project extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'title',
        'type',
        'research_team_id', // جایگزین institution_id
        'description',
        'owner_id',
        'status',
        'risk_level',
        'risk_analysis',
        'sku',
        'category',
        'custom_category',
        'start_date',
        'approval_date',
        'end_date',
        'budget',
        'warehouse_manager_id',
    ];
    /**
     * Get the route key for the model.
     * این متد به لاراول می‌گوید که از ستون 'sku' برای پیدا کردن مدل در URL استفاده کند.
     *
     * @return string
     */
    public function getRouteKeyName()
    {
        return 'sku';
    }

    /**
     * Get the research team that the project belongs to.
     */
    public function researchTeam(): BelongsTo
    {
        return $this->belongsTo(ResearchTeam::class);
    }

//    /**
//     * Get the institution that the project belongs to.
//     */
//    public function institution(): BelongsTo
//    {
//        return $this->belongsTo(Institution::class);
//    }

    /**
     * Get the user who owns the project.
     */
    public function owner(): BelongsTo
    {
        return $this->belongsTo(User::class, 'owner_id');
    }

    /**
     * The users that belong to the project as collaborators.
     */
    public function collaborators(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'project_user', 'project_id', 'user_id')
            ->withPivot('project_role')
            ->withTimestamps();
    }

    /**
     * [NEW] Helper method to get a specific user's role in this project.
     *
     * @param User $user
     * @return string|null
     */
    public function getUserRole(User $user): ?string
    {
        // اگر کاربر مالک باشد، نقش "مالک طرح" را برمی‌گردانیم
        if ($this->owner_id === $user->id) {
            return 'project_owner';
        }

        $collaborator = $this->collaborators()->where('user_id', $user->id)->first();

        return $collaborator?->pivot?->project_role;
    }

    /**
     * The protocols that belong to the project.
     */
    public function protocols(): BelongsToMany
    {
        return $this->belongsToMany(protocols::class, 'project_protocol', 'project_id', 'protocol_id')
            ->withTimestamps();
    }

    /**
     * Get all of the tags for the project.
     */
    public function tags(): MorphToMany
    {
        return $this->morphToMany(Tag::class, 'taggable');
    }

    /**
     * Get the lab notebook entries for the project.
     * این متد جدید، یادداشت‌های این طرح را مشخص می‌کند.
     */
    public function labNotebookEntries(): HasMany
    {
        return $this->hasMany(LabNotebookEntry::class);
    }

    /**
     * تعریف رابطه چند به چند با آیتم‌های انبار برای بودجه‌بندی.
     * این رابطه از جدول واسط 'project_inventory_item' استفاده می‌کند.
     */
    public function budgetItems()
    {
        return $this->belongsToMany(InventoryItem::class, 'project_inventory_item')
            ->withPivot('max_allowed_quantity', 'total_consumed_quantity', 'id')
            ->withTimestamps()
            ->using(ProjectInventoryItem::class);
    }

    /**
     * [NEW] The permissions that have been granted to users specifically for this project.
     * این رابطه، تمام دسترسی‌های سفارشی که در این طرح به کاربران داده شده را برمی‌گرداند.
     */
    public function specificPermissions(): BelongsToMany
    {
        return $this->belongsToMany(Permission::class, 'project_user_permission', 'project_id', 'permission_id')
            ->withPivot('user_id')
            ->withTimestamps();
    }
    /**
     * [NEW] Get the user designated as the warehouse manager for this project.
     */
    public function warehouseManager(): BelongsTo
    {
        return $this->belongsTo(User::class, 'warehouse_manager_id');
    }

    /**
     * [NEW] Get the tasks for the project.
     */
    public function tasks(): HasMany
    {
        return $this->hasMany(Task::class);
    }
    /**
     * [NEW] Get the announcements for the project.
     */
    public function announcements(): HasMany
    {
        return $this->hasMany(ProjectAnnouncement::class)->latest();
    }
    /**
     * [NEW] Get the chat room associated with the project.
     */
    public function chatRoom(): HasOne
    {
        return $this->hasOne(ChatRoom::class);
    }
}
