<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Spatie\Permission\Models\Permission;

class ResearchTeam extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'description',
        'research_teams_code',
        'institution_id',
        'calibration_manager_id',
        'purchasing_manager_id',
        'warehouse_manager_id',
    ];

    public function institution(): BelongsTo
    {
        return $this->belongsTo(Institution::class);
    }

    /**
     * Get the inventory items for the research team.
     */
    public function inventoryItems(): HasMany
    {
        return $this->hasMany(InventoryItem::class);
    }

    public function users(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'research_team_user')->withPivot('role')->withTimestamps();
    }
    /**
     * [NEW] Get only the users who are admins of this team.
     */
    public function managers(): BelongsToMany
    {
        return $this->users()->wherePivot('role', 'admin');
    }
    /**
     * [NEW] Get the equipment for the research team.
     */
    public function equipment(): HasMany
    {
        return $this->hasMany(Equipment::class);
    }
    /**
     * [NEW] Get the user assigned as the calibration manager for this team.
     */
    public function calibrationManager(): BelongsTo
    {
        return $this->belongsTo(User::class, 'calibration_manager_id');
    }
    /**
     * [NEW] Get the user assigned as the purchasing manager for this team.
     */
    public function purchasingManager(): BelongsTo
    {
        return $this->belongsTo(User::class, 'purchasing_manager_id');
    }

    /**
     * [REFACTORED] Get the user assigned as the warehouse manager for this team using a direct relationship.
     */
    public function warehouseManager(): BelongsTo
    {
        return $this->belongsTo(User::class, 'warehouse_manager_id');
    }

    /**
     * [NEW] Get the pending invitations for the team.
     */
    public function invitations(): HasMany
    {
        return $this->hasMany(TeamInvitation::class, 'team_id');
    }

    public function projects(): HasMany
    {
        return $this->hasMany(Project::class);
    }
    /**
     * [NEW] Get the announcements for the research team.
     */
    public function announcements(): HasMany
    {
        return $this->hasMany(TeamAnnouncement::class)->latest();
    }
    /**
     * [NEW] Get the onboarding checklist items for the research team.
     */
    public function onboardingChecklistItems(): HasMany
    {
        return $this->hasMany(OnboardingChecklistItem::class)->orderBy('order');
    }
    /**
     * [NEW] The permissions that have been granted to users specifically for this team.
     */
    public function permissions(): BelongsToMany
    {
        return $this->belongsToMany(Permission::class, 'research_team_user_permission', 'research_team_id', 'permission_id')
            ->withPivot('user_id')
            ->withTimestamps();
    }

    public function labNotebookTemplates(): HasMany
    {
        return $this->hasMany(LabNotebookTemplate::class);
    }
}

