<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class StorageLocation extends Model
{
    use HasFactory;
    /**
     * [NEW] The "booted" method of the model.
     * This ensures that when a location is deleted, all its children are deleted first.
     */
    protected static function booted(): void
    {
        static::deleting(function (StorageLocation $location) {
            // Recursively delete all children before deleting the parent
            foreach ($location->children as $child) {
                $child->delete();
            }
        });
    }

    protected $fillable = [
        'name',
        'location_code',
        'location_type',
        'research_team_id',
        'parent_id',
        'path',
        'rows',
        'columns',
        'internal_structure_type',
        'uuid',
    ];

    /**
     * Get the parent for the location.
     */
    public function parent(): BelongsTo
    {
        return $this->belongsTo(StorageLocation::class, 'parent_id');
    }

    /**
     * Get the children for the location.
     */
    public function children(): HasMany
    {
        return $this->hasMany(StorageLocation::class, 'parent_id');
    }

    /**
     * Get the research team that owns the location.
     */
    public function researchTeam(): BelongsTo
    {
        return $this->belongsTo(ResearchTeam::class);
    }

    public function inventoryItems(): HasMany
    {
        return $this->hasMany(InventoryItem::class);
    }

    /**
     * Get the full path of the location.
     * این یک Accessor است که مسیر کامل را به صورت خودکار ایجاد می‌کند.
     */
    public function getFullPathAttribute(): string
    {
        $path = $this->name;
        $parent = $this->parent;

        while ($parent) {
            $path = $parent->name . ' > ' . $path;
            $parent = $parent->parent;
        }

        return $path;
    }

    /**
     * [NEW] Recursively checks if the current location is a descendant of a location with a specific type.
     *
     * @param string $type The type to check for (e.g., 'tank').
     * @return bool
     */
    public function isInside(string $type): bool
    {
        $parent = $this->parent;
        while ($parent) {
            if ($parent->location_type === $type) {
                return true;
            }
            $parent = $parent->parent;
        }
        return false;
    }
}
