<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Notifications\Notifiable;
use Laravel\Fortify\TwoFactorAuthenticatable;
use Laravel\Jetstream\HasProfilePhoto;
use Laravel\Jetstream\HasTeams;
use Laravel\Sanctum\HasApiTokens;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Spatie\Permission\Traits\HasRoles; // <-- قدم ۱: ایمپورت کردن Trait
use Spatie\Permission\Models\Permission;




class User extends Authenticatable
{
    use HasApiTokens;
    use HasFactory;
    use HasProfilePhoto;
    use HasTeams;
    use Notifiable;
    use TwoFactorAuthenticatable;
    use HasRoles; // <-- قدم ۲: استفاده از Trait


    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'first_name',
        'last_name',
        'username',
        'national_id',
        'email',
        'phone_number',
        'postal_address',
        'password',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
        'two_factor_recovery_codes',
        'two_factor_secret',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
    ];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array<int, string>
     */
    protected $appends = [
        'profile_photo_url',
    ];

    /**
     * The system-wide roles that belong to the user.
     */
    public function systemRoles(): BelongsToMany
    {
        return $this->belongsToMany(Role::class, 'role_user', 'user_id', 'role_id')->withTimestamps();
    }

    /**
     * Check if the user has a specific system-wide role.
     *
     * @param string $roleName
     * @return bool
     */
//    public function hasSystemRole(string $roleName): bool
//    {
//        return $this->systemRoles()->where('name', $roleName)->exists();
//    }
    /**
     * [NEW] The projects that the user owns.
     */
    public function projectsAsOwner(): HasMany
    {
        return $this->hasMany(Project::class, 'owner_id');
    }
    /**
     * The projects that the user is a collaborator on.
     * این متد جدید، لیست پروژه‌هایی را که این کاربر در آن‌ها همکار است، برمی‌گرداند.
     */
    public function projectsAsCollaborator(): BelongsToMany
    {
        return $this->belongsToMany(Project::class, 'project_user', 'user_id', 'project_id')
            ->withPivot('project_role')
            ->withTimestamps();
    }

    /**
     * The research teams that the user belongs to.
     */
    public function researchTeams(): BelongsToMany
    {
        return $this->belongsToMany(ResearchTeam::class, 'research_team_user')->withPivot('role')->withTimestamps();
    }

    public function institutionMemberships(): HasMany
    {
        return $this->hasMany(InstitutionMembership::class);
    }
    public function lowStockAcknowledgements(): HasMany
    {
        return $this->hasMany(LowStockAcknowledgement::class);
    }

    public function institutionRequests(): HasMany
    {
        return $this->hasMany(InstitutionRequest::class);
    }

    /**
     * [NEW] The permissions that a user has been directly granted within a specific project.
     * این رابطه، تمام دسترسی‌های سفارشی یک کاربر در تمام طرح‌ها را برمی‌گرداند.
     */
    public function projectPermissions(): BelongsToMany
    {
        return $this->belongsToMany(Permission::class, 'project_user_permission', 'user_id', 'permission_id')
            ->withPivot('project_id')
            ->withTimestamps();
    }

    /**
     * [NEW] Helper method to check if a user has a specific permission directly in a given project.
     * این متد بررسی می‌کند که آیا کاربر به صورت مستقیم، یک دسترسی خاص را در یک طرح مشخص دارد یا خیر.
     *
     * @param string|Permission $permission
     * @param Project $project
     * @return bool
     */
    public function hasDirectPermissionInProject($permission, Project $project): bool
    {
        $permissionName = is_string($permission) ? $permission : $permission->name;

        return $this->projectPermissions()
            ->where('project_id', $project->id)
            ->where('name', $permissionName)
            ->exists();
    }
    /**
     * [NEW] Get the user's full name.
     *
     * @return string
     */
    public function getNameAttribute(): string
    {
        return "{$this->first_name} {$this->last_name}";
    }

    /**
     * [NEW] The tasks that are assigned to the user.
     */
    public function assignedTasks(): BelongsToMany
    {
        return $this->belongsToMany(Task::class, 'task_user', 'user_id', 'task_id');
    }

    /**
     * [NEW] The tasks that were created by the user.
     */
    public function createdTasks(): HasMany
    {
        return $this->hasMany(Task::class, 'creator_id');
    }
    /**
     * [NEW] Get the time logs for the user.
     */
    public function timeLogs(): HasMany
    {
        return $this->hasMany(TimeLog::class);
    }
    /**
     * [NEW] Get the project announcements created by the user.
     */
    public function projectAnnouncements(): HasMany
    {
        return $this->hasMany(ProjectAnnouncement::class);
    }
    /**
     * [NEW] The chat rooms that the user belongs to.
     */
    public function chatRooms(): BelongsToMany
    {
        return $this->belongsToMany(ChatRoom::class, 'chat_room_user', 'user_id', 'chat_room_id');
    }

    /**
     * [NEW] Get the chat messages for the user.
     */
    public function chatMessages(): HasMany
    {
        return $this->hasMany(ChatMessage::class);
    }
    /**
     * [NEW] The chat messages that the user has read.
     */
    public function readChatMessages(): BelongsToMany
    {
        return $this->belongsToMany(ChatMessage::class, 'chat_message_read_status', 'user_id', 'chat_message_id')
            ->withPivot('read_at');
    }
    /**
     * [NEW] The equipment that the user is responsible for.
     */
    public function equipmentResponsibilities(): HasMany
    {
        return $this->hasMany(Equipment::class, 'responsible_user_id');
    }

    /**
     * [NEW] Get the calibration logs created by the user.
     */
    public function calibrationLogs(): HasMany
    {
        return $this->hasMany(CalibrationLog::class);
    }
    /**
     * [NEW] Get the team announcements created by the user.
     */
    public function teamAnnouncements(): HasMany
    {
        return $this->hasMany(TeamAnnouncement::class);
    }
    /**
     * [NEW] The permissions that a user has been directly granted within a specific team.
     */
    public function teamPermissions(): BelongsToMany
    {
        return $this->belongsToMany(Permission::class, 'research_team_user_permission', 'user_id', 'permission_id')
            ->withPivot('research_team_id')
            ->withTimestamps();
    }

    /**
     * [NEW] Helper method to check if a user has a specific permission in a given team.
     */
    public function hasTeamPermission(ResearchTeam $team, string $permissionName): bool
    {
        // 1. ابتدا نقش کاربر در تیم را بررسی می‌کنیم
        $roleInTeam = $team->users()->where('user_id', $this->id)->first()?->pivot->role;

        // اگر مدیر تیم باشد، به همه چیز دسترسی دارد
        if ($roleInTeam === 'admin') {
            return true;
        }

        // 2. سپس، مجوزهای مستقیم کاربر در آن تیم را بررسی می‌کنیم
        return $this->teamPermissions()
            ->where('research_team_id', $team->id)
            ->where('name', $permissionName)
            ->exists();
    }
    /**
     * [NEW] Route notifications for the RayganSms channel.
     *
     * @param  \Illuminate\Notifications\Notification  $notification
     * @return string
     */
    public function routeNotificationForRayganSms($notification)
    {
        // شماره تلفن کاربر را از ستون phone_number برمی‌گرداند
        return $this->phone_number;
    }
    public function comments(): HasMany
    {
        // This defines that a user can have many comments.
        return $this->hasMany(LabNotebookComment::class);
    }

    /**
     * [NEW] Get the personal notes for the user.
     */
    public function personalNotes(): HasMany
    {
        return $this->hasMany(PersonalNote::class);
    }
}
