<?php

namespace App\Notifications;

use App\Models\Equipment;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class CalibrationDueNotification extends Notification
{
    use Queueable;

    public Equipment $equipment;
    public int $daysRemaining;

    /**
     * Create a new notification instance.
     */
    public function __construct(Equipment $equipment, int $daysRemaining)
    {
        $this->equipment = $equipment;
        $this->daysRemaining = $daysRemaining;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $equipmentUrl = route('equipment.index'); // لینک به صفحه مدیریت تجهیزات
        $subject = $this->daysRemaining > 0
            ? 'یادآوری: ' . $this->daysRemaining . ' روز تا سررسید کالیبراسیون تجهیز "' . $this->equipment->name . '" باقی مانده است'
            : 'یادآوری: امروز سررسید کالیبراسیون تجهیز "' . $this->equipment->name . '" است';

        return (new MailMessage)
            ->subject($subject)
            ->greeting('سلام ' . $notifiable->name . ',')
            ->line('این یک یادآوری برای کالیبراسیون تجهیز زیر است:')
            ->line('**نام تجهیز:** ' . $this->equipment->name)
            ->line('**تیم تحقیقاتی:** ' . $this->equipment->researchTeam?->name)
            ->line('**تاریخ سررسید:** ' . verta($this->equipment->next_calibration_date)->format('Y/m/d'))
            ->action('مشاهده لیست تجهیزات', $equipmentUrl);
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        $message = $this->daysRemaining > 0
            ? $this->daysRemaining . ' روز تا سررسید کالیبراسیون تجهیز "' . $this->equipment->name . '" باقی مانده است.'
            : 'امروز سررسید کالیبراسیون تجهیز "' . $this->equipment->name . '" است.';

        return [
            'icon' => 'timer', // نام آیکون از Lucide Icons
            'title' => 'یادآوری سررسید کالیبراسیون',
            'message' => $message,
            'url' => route('equipment.index'),
        ];
    }
}
