<?php

namespace App\Notifications;

use App\Models\Task;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class TaskDueReminderNotification extends Notification
{
    use Queueable;

    public Task $task;
    public int $daysRemaining;

    /**
     * Create a new notification instance.
     */
    public function __construct(Task $task, int $daysRemaining)
    {
        $this->task = $task;
        $this->daysRemaining = $daysRemaining;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $projectUrl = route('projects.show', $this->task->project);
        $subject = $this->daysRemaining > 0
            ? 'یادآوری: ' . $this->daysRemaining . ' روز تا پایان مهلت وظیفه "' . $this->task->title . '" باقی مانده است'
            : 'یادآوری: امروز آخرین مهلت انجام وظیفه "' . $this->task->title . '" است';

        return (new MailMessage)
            ->subject($subject)
            ->greeting('سلام ' . $notifiable->name . ',')
            ->line('این یک یادآوری برای وظیفه زیر است:')
            ->line('**عنوان وظیفه:** ' . $this->task->title)
            ->line('**طرح:** ' . $this->task->project->title)
            ->line('**تاریخ پایان:** ' . verta($this->task->due_date)->format('Y/m/d'))
            ->action('مشاهده جزئیات طرح', $projectUrl);
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        $message = $this->daysRemaining > 0
            ? $this->daysRemaining . ' روز تا پایان مهلت وظیفه "' . $this->task->title . '" باقی مانده است.'
            : 'امروز آخرین مهلت انجام وظیفه "' . $this->task->title . '" است.';

        return [
            'icon' => 'clock', // نام آیکون از Lucide Icons
            'title' => 'یادآوری مهلت وظیفه',
            'message' => $message,
            'url' => route('projects.show', $this->task->project),
        ];
    }
}
