<?php

namespace App\Policies;

use App\Models\InventoryItem;
use App\Models\User;
use Illuminate\Auth\Access\Response;

class InventoryItemPolicy
{
    /**
     * Perform pre-authorization checks.
     */
    public function before(User $user, string $ability): bool|null
    {
        // اگر کاربر سوپر-ادمین باشد، به همه چیز دسترسی دارد.
        if ($user->hasRole('super_admin')) {
            return true;
        }

        return null; // در غیر این صورت، به سراغ بررسی متد مربوطه برو.
    }

    /**
     * Determine whether the user can view any models.
     * این متد مشخص می‌کند چه کسی اجازه دیدن لیست اقلام انبار را دارد.
     */
    public function viewAny(User $user): bool
    {

        // [REFACTORED] Logic to handle team-specific warehouse manager roles.

        // 1. Users with general roles that should always have access.
        if ($user->hasAnyRole(['monitoring_user', 'institution_admin', 'team_manager'])) {
            return true;
        }

        // 2. Check if the user is a warehouse manager in ANY of their teams.
        foreach ($user->researchTeams as $team) {
            if ($user->hasRole("Warehouse_Manager-Team-{$team->id}")) {
                return true; // Grant access if they have the role in at least one team.
            }
        }

        // 3. If none of the above, deny access.
        return false;
    }
    /**
     * Determine whether the user can create models.
     */
    public function create(User $user): bool
    {
        // The logic for creating can be the same as viewing the list.
        return $this->viewAny($user);
    }

    /**
     * Determine whether the user can update the model.
     */
    public function update(User $user, InventoryItem $inventoryItem): bool
    {
        // A user can update an item if:
        // 1. They are the owner of the item.
        if ($user->id === $inventoryItem->owner_id) {
            return true;
        }

        // 2. They are the warehouse manager for the team the item belongs to.
        if ($inventoryItem->research_team_id) {
            return $user->hasRole("Warehouse_Manager-Team-{$inventoryItem->research_team_id}");
        }

        return false;
    }

    /**
     * Determine whether the user can delete the model.
     */
    public function delete(User $user, InventoryItem $inventoryItem): bool
    {
        // A user can delete an item if:
        // 1. They are the owner of the item.
        if ($user->id === $inventoryItem->owner_id) {
            return true;
        }

        // 2. They are the warehouse manager for the team the item belongs to.
        if ($inventoryItem->research_team_id) {
            return $user->hasRole("Warehouse_Manager-Team-{$inventoryItem->research_team_id}");
        }

        return false;
    }

//    /**
//     * Determine whether the user can create models.
//     */
//    public function create(User $user): bool
//    {
//        // تمام نقش‌های اصلی به جز بازرگان می‌توانند آیتم جدید اضافه کنند.
//        return $user->hasRole('monitoring_user')
//            || $user->hasRole('institution_admin')
//            || $user->hasRole('researcher')
//            || $user->hasRole('student')
//            || $user->hasRole('Warehouse_Manager');
//    }
//
//    /**
//     * Determine whether the user can update the model.
//     */
//    public function update(User $user, InventoryItem $inventoryItem): bool
//    {
//        // فقط مالک آیتم می‌تواند آن را ویرایش کند.
//        return $user->id === $inventoryItem->owner_id;
//    }
//
//    /**
//     * Determine whether the user can delete the model.
//     */
//    public function delete(User $user, InventoryItem $inventoryItem): bool
//    {
//        // فقط مالک آیتم می‌تواند آن را حذف کند.
//        return $user->id === $inventoryItem->owner_id;
//    }
}
