<?php

namespace App\Policies;

use App\Models\ActivityRequest;
use App\Models\LabNotebookEntry;
use App\Models\User;
use Illuminate\Auth\Access\Response;
use App\Models\Project;

class LabNotebookEntryPolicy
{
    /**
     * Perform pre-authorization checks.
     */
    public function before(User $user, string $ability): bool|null
    {
        if ($user->hasRole('super_admin') || $user->hasRole('monitoring_user')) {
            return true;
        }
        return null;
    }

    /**
     * Determine whether the user can sign the entry.
     * این متد جدید، قانون دسترسی برای امضای یک یادداشت را تعریف می‌کند.
     */
    public function sign(User $user, LabNotebookEntry $entry): bool
    {
//        // اگر یادداشت از قبل قفل شده باشد، کسی نمی‌تواند آن را امضا کند.
//        if ($entry->is_locked) {
//            return false;
//        }
//
//        // مالک طرح همیشه می‌تواند امضا کند.
//        if ($user->id === $entry->project->owner_id) {
//            return true;
//        }
//
//        // یا کاربری که یک درخواست امضای فعال برای این یادداشت دارد.
//        return ActivityRequest::where('assignee_id', $user->id)
//            ->where('lab_notebook_entry_id', $entry->id)
//            ->where('activity_type', 'امضا')
//            ->where('status', 'pending')
//            ->exists();
        if ($entry->is_locked) {
            return false;
        }
        if ($user->id === $entry->project->owner_id) {
            return true;
        }
        // بررسی دسترسی مستقیم برای امضا
        if ($user->hasDirectPermissionInProject('labnotebook-sign', $entry->project)) {
            return true;
        }
        // یا اگر یک درخواست امضای فعال برای او وجود داشته باشد
        return ActivityRequest::where('assignee_id', $user->id)
            ->where('lab_notebook_entry_id', $entry->id)
            ->where('activity_type', 'امضا')
            ->where('status', 'pending')
            ->exists();
    }


    /**
     * Determine whether the user can update the model.
     */
    public function update(User $user, LabNotebookEntry $entry): bool
    {
//        if ($entry->is_locked) {
//            return false;
//        }
//        // نویسنده یادداشت، مالک طرح و همکار اصلی می‌توانند یادداشت را ویرایش کنند.
//        $userRoleInProject = $entry->project->getUserRole($user);
//        return $user->id === $entry->user_id || in_array($userRoleInProject, ['project_owner', 'main_collaborator']);
        if ($entry->is_locked) {
            return false;
        }
        // مالک طرح همیشه می‌تواند ویرایش کند
        if ($user->id === $entry->project->owner_id) {
            return true;
        }
        // نویسنده یادداشت نیز می‌تواند ویرایش کند
        if ($user->id === $entry->user_id) {
            return true;
        }
        // در غیر این صورت، بررسی دسترسی مستقیم
        return $user->hasDirectPermissionInProject('labnotebook-edit', $entry->project);
    }

    /**
     * Determine whether the user can manage materials for the entry.
     * این متد جدید، قانون دسترسی به مواد مصرفی را تعریف می‌کند.
     */
    public function manageMaterials(User $user, LabNotebookEntry $entry): bool
    {
//        // اگر یادداشت قفل شده باشد، هیچکس نمی‌تواند مواد را مدیریت کند.
//        if ($entry->is_locked) {
//            return false;
//        }
//        // در غیر این صورت، فقط مالک طرح می‌تواند مواد را مدیریت کند.
////        return $user->id === $entry->project->owner_id;
//        $userRoleInProject = $entry->project->getUserRole($user);
//        return in_array($userRoleInProject, ['project_owner', 'main_collaborator']);
        if ($entry->is_locked) {
            return false;
        }
        if ($user->id === $entry->project->owner_id) {
            return true;
        }
        return $user->hasDirectPermissionInProject('labnotebook-manage-materials', $entry->project);
    }


    /**
     * Determine whether the user can toggle the lock on the model.
     */
    public function toggleLock(User $user, LabNotebookEntry $entry): bool
    {
//        return $user->id === $entry->project->owner_id;
//        return $entry->project->getUserRole($user) === 'project_owner';
        if ($user->id === $entry->project->owner_id) {
            return true;
        }
        return $user->hasDirectPermissionInProject('labnotebook-lock', $entry->project);

    }

    /**
     * Determine whether the user can delete the model.
     */
    public function delete(User $user, LabNotebookEntry $entry): bool
    {
//        if ($entry->is_locked) {
//            return false;
//        }
//        return $user->id === $entry->user_id;
        if ($entry->is_locked) {
            return false;
        }
        // فقط نویسنده اصلی یادداشت می‌تواند آن را حذف کند (برای امنیت)
        return $user->id === $entry->user_id;
    }

    /**
     * [NEW] Determine whether the user can create models.
     * این متد مشخص می‌کند چه کسی می‌تواند دکمه "افزودن یادداشت جدید" را ببیند.
     */
    public function create(User $user, Project $project): bool
    {
        // مالک طرح همیشه می‌تواند یادداشت جدید ایجاد کند.
        if ($user->id === $project->owner_id) {
            return true;
        }
        // در غیر این صورت، بررسی می‌کنیم آیا دسترسی مستقیم دارد یا خیر.
        return $user->hasDirectPermissionInProject('labnotebook-create', $project);
    }
}
