<?php

namespace App\Policies;

use App\Models\Project;
use App\Models\User;
use Illuminate\Auth\Access\Response;
use Illuminate\Contracts\Auth\Authenticatable; // این خط را در بالای فایل اضافه کنید


class ProjectPolicy
{
    /**
     * Perform pre-authorization checks.
     */
    public function before(User $user, string $ability): bool|null
    {
//        if ($user->hasSystemRole('super_admin')) {
//            return true;
            if ($user->hasRole('super_admin')) {
            return true;
        }

        return null;
    }



    /**
     * Determine whether the user can view the model.
     * این متد جدید، قانون دسترسی برای مشاهده جزئیات یک طرح را تعریف می‌کند.
     */
    public function view(User $user, Project $project): bool
    {
        // کاربر می‌تواند طرح را ببیند اگر مالک آن باشد
        if ($user->id === $project->owner_id) {
            return true;
        }

        // یا اگر یکی از همکاران طرح باشد
        return $project->collaborators->contains($user);
    }

    /**
     * Determine whether the user can view any models.
     * این متد مشخص می‌کند چه کسی اجازه دیدن لینک و صفحه اصلی "مدیریت طرح و برنامه" را دارد.
     */
    public function viewAny(User $user): bool
    {
        // فقط نقش‌های مدیریتی و تحقیقاتی می‌توانند این صفحه را ببینند.
        return $user->hasRole('monitoring_user')
            || $user->hasRole('institution_admin')
            || $user->hasRole('researcher')
            || $user->projectsAsCollaborator()->exists()
            || $user->hasRole('team_manager');

    }

    /**
     * Determine whether the user can create models.
     */
    public function create(User $user): bool
    {
        return $user->hasRole('researcher') || $user->hasRole('institution_admin') || $user->hasRole('team_manager');
    }

    /**
     * Determine whether the user can update the model.
     */
    public function update(User $user, Project $project): bool
    {
//        // فقط مالک طرح می‌تواند آن را ویرایش کند.
////        return $user->id === $project->owner_id;
//        $userRole = $project->getUserRole($user);
//        return $userRole === 'project_owner' || $userRole === 'main_collaborator';
        // مالک طرح همیشه می‌تواند ویرایش کند
        if ($user->id === $project->owner_id) {
            return true;
        }
        // در غیر این صورت، بررسی می‌کنیم آیا دسترسی مستقیم دارد یا خیر
        return $user->hasDirectPermissionInProject('project-edit', $project);
    }

    /**
     * Determine whether the user can delete the model.
     */
    public function delete(User $user, Project $project): bool
    {
        // فقط مالک طرح می‌تواند آن را حذف کند.
        return $user->id === $project->owner_id;
    }

    /**
     * [NEW] Determine whether the user can manage collaborators.
     */
    public function manageCollaborators(User $user, Project $project): bool
    {
        if ($user->id === $project->owner_id) {
            return true;
        }
        return $user->hasDirectPermissionInProject('project-manage-collaborators', $project);
    }

    /**
     * [NEW] Determine whether the user can manage the project budget.
     */
    public function manageBudget(User $user, Project $project): bool
    {
        // مالک طرح و همکار اصلی می‌توانند بودجه مواد را مدیریت کنند.
        if ($user->id === $project->owner_id) {
            return true;
        }
        return $user->hasDirectPermissionInProject('project-manage-budget', $project);
    }

    public function warehouseManager(User $user, Project $project): bool
    {
        // مالک طرح و همکار اصلی می‌توانند بودجه مواد را مدیریت کنند.
        if ($user->id === $project->owner_id) {
            return true;
        }
        return $user->hasDirectPermissionInProject('warehouse-manage-requests', $project);
    }
    /**
     * [NEW] Determine whether the user can view collaborator performance reports for the project.
     */
    public function viewCollaboratorReports(User $user, Project $project): bool
    {
        // فقط مالک طرح می‌تواند گزارش‌های عملکرد را مشاهده کند
        return $user->id === $project->owner_id;
    }

    public function viewProjectReport(User $user, Project $project): bool
    {
        // فقط مالک طرح می‌تواند گزارش‌های عملکرد را مشاهده کند
        return $user->id === $project->owner_id;
    }
}
