<?php

namespace App\Policies;

use App\Models\Project;
use App\Models\Task;
use App\Models\User;

class TaskPolicy
{
    /**
     * Perform pre-authorization checks.
     */
    public function before(User $user, string $ability): bool|null
    {
        if ($user->hasRole('super_admin')) {
            return true;
        }
        // مالک طرح به تمام وظایف طرح خود دسترسی کامل دارد
        if ($task = $this->getTaskFromAbility($ability)) {
            if ($user->id === $task->project->owner_id) {
                return true;
            }
        }

        return null;
    }

    /**
     * Determine whether the user can create models.
     */
    public function create(User $user, Project $project): bool
    {
        $userRole = $project->getUserRole($user);
        return in_array($userRole, ['project_owner', 'main_collaborator']);
    }

    /**
     * Determine whether the user can update the model.
     */
    public function update(User $user, Task $task): bool
    {
        $userRole = $task->project->getUserRole($user);
        // مالک طرح، ایجادکننده وظیفه یا یکی از مسئولین آن می‌توانند وظیفه را ویرایش کنند
        return in_array($userRole, ['project_owner']) || $user->id === $task->creator_id || $task->assignees->contains($user);
    }

    /**
     * Determine whether the user can delete the model.
     */
    public function delete(User $user, Task $task): bool
    {
        // فقط مالک طرح و ایجادکننده وظیفه می‌توانند آن را حذف کنند
        $userRole = $task->project->getUserRole($user);
        return in_array($userRole, ['project_owner']) || $user->id === $task->creator_id;
    }

    /**
     * Determine whether the user can change the status of the model.
     */
    public function changeStatus(User $user, Task $task): bool
    {
        // فقط کاربری که وظیفه به او محول شده می‌تواند وضعیت را تغییر دهد
        return $task->assignees->contains($user);
    }

    /**
     * Helper to get task instance when available in policy methods.
     */
    private function getTaskFromAbility(string $ability): ?Task
    {
        // This is a simplified helper. In a real app, you might need a more robust way
        // to get the context model if it's not passed directly.
        // For our current use case in the components, the model is passed directly.
        return null;
    }
}
