<?php
//
//namespace App\Services;
//
//use Illuminate\Support\Facades\Http;
//use Illuminate\Support\Facades\Log;
//
//class AIService
//{
//    protected string $apiKey;
//    protected string $apiUrl;
//
//    public function __construct()
//    {
//        $this->apiKey = config('services.gemini.api_key');
//        $this->apiUrl = 'https://generativelanguage.googleapis.com/v1beta/models/gemini-pro:generateContent';
//    }
//
//    /**
//     * Analyzes the given notebook text using the Gemini API.
//     *
//     * @param string $text The text content of the lab notebook entry.
//     * @return array|null The structured analysis result or null on failure.
//     */
//    public function analyzeNotebookEntry(string $text): ?array
//    {
//        if (empty($this->apiKey)) {
//            Log::error('Gemini API key is not configured.');
//            return null;
//        }
//
//        // We create a very specific prompt to get a structured JSON response.
//        $prompt = "Analyze the following lab notebook entry written in Persian. Provide a concise summary and extract key entities. The output MUST be a valid JSON object with two keys: 'summary' and 'entities'. The 'summary' should be a string. The 'entities' should be an array of objects, where each object has a 'type' (e.g., 'Material', 'Measurement', 'Result') and a 'value' (the extracted text). Here is the text:\n\n---\n\n{$text}";
//
//        try {
//            $response = Http::timeout(60)->post("{$this->apiUrl}?key={$this->apiKey}", [
//                'contents' => [
//                    [
//                        'parts' => [
//                            ['text' => $prompt]
//                        ]
//                    ]
//                ]
//            ]);
//
//            if ($response->failed()) {
//                Log::error('Gemini API request failed.', ['response' => $response->body()]);
//                return null;
//            }
//
//            // Extract the text content which should be a JSON string
//            $jsonString = $response->json('candidates.0.content.parts.0.text', '');
//
//            // Clean the response to ensure it's valid JSON
//            $cleanedJson = trim(str_replace(['```json', '```'], '', $jsonString));
//
//            $result = json_decode($cleanedJson, true);
//
//            if (json_last_error() !== JSON_ERROR_NONE) {
//                Log::error('Failed to decode JSON from Gemini API response.', ['raw_response' => $jsonString]);
//                return null;
//            }
//
//            return $result;
//
//        } catch (\Exception $e) {
//            Log::error('Exception occurred while calling Gemini API.', ['message' => $e->getMessage()]);
//            return null;
//        }
//    }
//}


namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;


class AIService
{
    protected string $apiKey;
    protected string $apiUrl;
    protected string $visionApiUrl; // [NEW]

    public function __construct()
    {
        // [FIXED] Updated to use DeepSeek configuration
//        $this->apiKey = config('services.deepseek.api_key');
        $this->apiKey = config('services.deepseek.api_key');
//        $this->apiUrl = 'https://api.deepseek.com/v1/chat/completions';
        $this->apiUrl = 'https://api.avalai.ir/v1/chat/completions';
        $this->visionApiUrl = 'https://api.avalai.ir/v1/chat/completions'; // Same endpoint for VL model

    }

    /**
     * Analyzes the given notebook text using the DeepSeek API.
     *
     * @param string $text The text content of the lab notebook entry.
     * @return array|null The structured analysis result or null on failure.
     */
    public function analyzeNotebookEntry(string $text): ?array
    {
        if (empty($this->apiKey)) {
            Log::error('DeepSeek API key is not configured.');
            return null;
        }

        // System prompt to instruct the model for a structured JSON output
        $systemPrompt = "You are an expert scientific assistant. Analyze the provided lab notebook entry written in Persian. Your task is to provide a concise summary and extract key entities. The output MUST be a valid JSON object with two keys: 'summary' and 'entities'. The 'summary' should be a string in Persian. The 'entities' should be an array of objects, where each object has a 'type' (e.g., 'Material', 'Measurement', 'Result') and a 'value' (the extracted text). Do not include any text outside of the JSON object.";

        try {
            // [FIXED] Updated payload structure and authentication for DeepSeek API
            $response = Http::withoutVerifying() // Kept for local SSL issue
            ->withToken($this->apiKey)
                ->timeout(60)
                ->post($this->apiUrl, [
                    'model' => 'deepseek-chat',
                    'messages' => [
                        ['role' => 'system', 'content' => $systemPrompt],
                        ['role' => 'user', 'content' => $text]
                    ],
                    'stream' => false,
                ]);

            if ($response->failed()) {
                Log::error('DeepSeek API request failed.', ['response' => $response->body()]);
                return null;
            }

            // [FIXED] Extract the text content from DeepSeek's response structure
            $jsonString = $response->json('choices.0.message.content', '');

            $cleanedJson = trim(str_replace(['```json', '```'], '', $jsonString));
            $result = json_decode($cleanedJson, true);

            if (json_last_error() !== JSON_ERROR_NONE) {
                Log::error('Failed to decode JSON from DeepSeek API response.', ['raw_response' => $jsonString]);
                return null;
            }

            return $result;

        } catch (\Exception $e) {
            Log::error('Exception occurred while calling DeepSeek API.', ['message' => $e->getMessage()]);
            return null;
        }
    }

    /**
     * [NEW] Analyzes an image for cell counting using the DeepSeek Vision API.
     *
     * @param string $imagePath The storage path of the image file.
     * @return array|null The structured analysis result or null on failure.
     */
    public function analyzeImageForCellCount(string $imagePath): ?array
    {
        if (empty($this->apiKey) || !Storage::disk('public')->exists($imagePath)) {
            Log::error('DeepSeek API key is not configured or image not found.', ['path' => $imagePath]);
            return null;
        }

        // Read image data and encode it to base64
        $imageData = base64_encode(Storage::disk('public')->get($imagePath));
        $mimeType = Storage::disk('public')->mimeType($imagePath);
        $base64Image = "data:{$mimeType};base64,{$imageData}";

        $prompt = "You are an expert biologist assistant specializing in image analysis. Analyze the provided microscopy image. Your task is to count the number of distinct cells visible. Provide the result as a valid JSON object with two keys: 'cell_count' (an integer) and 'observation' (a brief string in Persian describing the visual characteristics, e.g., 'Cells are clustered', 'Uniform distribution'). Do not include any text outside of the JSON object.";

        try {
            $response = Http::withoutVerifying()
                ->withToken($this->apiKey)
                ->timeout(120) // Increased timeout for image processing
                ->post($this->visionApiUrl, [
                    'model' => 'deepseek-vl-chat',
                    'messages' => [
                        [
                            'role' => 'user',
                            'content' => [
                                ['type' => 'text', 'text' => $prompt],
                                ['type' => 'image_url', 'image_url' => ['url' => $base64Image]]
                            ]
                        ]
                    ]
                ]);

            if ($response->failed()) {
                Log::error('DeepSeek Vision API request failed.', ['response' => $response->body()]);
                return null;
            }

            $jsonString = $response->json('choices.0.message.content', '');
            $cleanedJson = trim(str_replace(['```json', '```'], '', $jsonString));
            $result = json_decode($cleanedJson, true);

            if (json_last_error() !== JSON_ERROR_NONE) {
                Log::error('Failed to decode JSON from DeepSeek Vision API.', ['raw_response' => $jsonString]);
                return null;
            }

            return $result;

        } catch (\Exception $e) {
            Log::error('Exception occurred while calling DeepSeek Vision API.', ['message' => $e->getMessage()]);
            return null;
        }
    }

    /**
     * [NEW] A generic method to send a prompt and get a structured JSON response.
     *
     * @param string $prompt The detailed prompt for the AI.
     * @return array|null The decoded JSON response or null on failure.
     */
    public function getStructuredResponse(string $prompt): ?array
    {
        if (empty($this->apiKey)) {
            Log::error('DeepSeek API key is not configured.');
            return null;
        }

        try {
            $response = Http::withoutVerifying()
                ->withToken($this->apiKey)
                ->timeout(60)
                ->post($this->apiUrl, [
                    'model' => 'deepseek-chat',
                    'messages' => [
                        ['role' => 'system', 'content' => 'You are a helpful assistant that only responds with valid JSON.'],
                        ['role' => 'user', 'content' => $prompt]
                    ],
                    'stream' => false,
                ]);

            if ($response->failed()) {
                Log::error('DeepSeek API request failed.', ['response' => $response->body()]);
                return null;
            }

            $jsonString = $response->json('choices.0.message.content', '');
            $cleanedJson = trim(str_replace(['```json', '```'], '', $jsonString));
            $result = json_decode($cleanedJson, true);

            if (json_last_error() !== JSON_ERROR_NONE) {
                Log::error('Failed to decode JSON from DeepSeek API response.', ['raw_response' => $jsonString]);
                return null;
            }

            return $result;

        } catch (\Exception $e) {
            Log::error('Exception occurred while calling DeepSeek API.', ['message' => $e->getMessage()]);
            return null;
        }
    }
}
