<?php
//
//namespace App\Services;
//
//use App\Models\Payment;
//use Illuminate\Support\Facades\Http;
//use Illuminate\Support\Facades\Log;
//
//class PaymentService
//{
//    protected string $merchantId;
//    protected bool $isSandbox;
//    protected string $requestUrl;
//    protected string $verifyUrl;
//    protected string $redirectUrl;
//
//    public function __construct()
//    {
//        $this->merchantId = config('services.zarinpal.merchant_id');
//        $this->isSandbox = config('services.zarinpal.sandbox', false);
//
//        if ($this->isSandbox) {
//            $this->requestUrl = 'https://sandbox.zarinpal.com/pg/v4/payment/request.json';
//            $this->verifyUrl = 'https://sandbox.zarinpal.com/pg/v4/payment/verify.json';
//            $this->redirectUrl = 'https://sandbox.zarinpal.com/pg/StartPay/';
//        } else {
//            $this->requestUrl = 'https://api.zarinpal.com/pg/v4/payment/request.json';
//            $this->verifyUrl = 'https://api.zarinpal.com/pg/v4/payment/verify.json';
//            $this->redirectUrl = 'https://www.zarinpal.com/pg/StartPay/';
//        }
//    }
//
//    /**
//     * Request a payment URL from Zarinpal.
//     *
//     * @param Payment $payment
//     * @return string|null The redirect URL or null on failure.
//     */
//    public function requestPayment(Payment $payment): ?string
//    {
//        if (empty($this->merchantId)) {
//            Log::error('Zarinpal Merchant ID is not configured.');
//            return null;
//        }
//
//        try {
//            $response = Http::post($this->requestUrl, [
//                'merchant_id' => $this->merchantId,
//                'amount' => $payment->amount * 10, // Converting Toman to Rial
//                'callback_url' => route('payment.callback', ['uuid' => $payment->uuid]),
//                'description' => 'پرداخت برای درخواست ' . $payment->payable->purchaseRequest->request_code,
//                'metadata' => [
//                    'email' => $payment->user->email,
//                ],
//            ]);
//
//            if ($response->successful() && $response->json('data.code') === 100) {
//                $authority = $response->json('data.authority');
//                $payment->update(['transaction_id' => $authority]); // Storing Authority as transaction_id
//                return $this->redirectUrl . $authority;
//            } else {
//                Log::error('Zarinpal request failed.', ['response' => $response->body()]);
//                return null;
//            }
//        } catch (\Exception $e) {
//            Log::error('Exception during Zarinpal payment request.', ['message' => $e->getMessage()]);
//            return null;
//        }
//    }
//
//    /**
//     * Verify the payment after the user returns from Zarinpal.
//     *
//     * @param array $callbackData Data from the callback URL (e.g., request()->all())
//     * @return Payment|null The updated payment record or null on failure.
//     */
//    public function verifyPayment(array $callbackData): ?Payment
//    {
//        $authority = $callbackData['Authority'] ?? null;
//        $status = $callbackData['Status'] ?? null;
//
//        if (!$authority || !$status) {
//            return null;
//        }
//
//        $payment = Payment::where('transaction_id', $authority)->firstOrFail();
//
//        if ($status !== 'OK') {
//            $payment->update(['status' => 'failed']);
//            return $payment;
//        }
//
//        try {
//            $response = Http::post($this->verifyUrl, [
//                'merchant_id' => $this->merchantId,
//                'amount' => $payment->amount * 10, // Converting Toman to Rial
//                'authority' => $authority,
//            ]);
//
//            if ($response->successful() && $response->json('data.code') === 100) {
//                $payment->update([
//                    'status' => 'successful',
//                    'paid_at' => now(),
//                    'meta' => ['ref_id' => $response->json('data.ref_id')],
//                ]);
//                return $payment;
//            } else {
//                // Payment was successful but verification failed (e.g., already verified)
//                // It's important to handle different error codes from Zarinpal here.
//                $payment->update(['status' => 'failed', 'meta' => $response->json('errors')]);
//                Log::error('Zarinpal verification failed.', ['response' => $response->body()]);
//                return $payment;
//            }
//        } catch (\Exception $e) {
//            Log::error('Exception during Zarinpal payment verification.', ['message' => $e->getMessage()]);
//            $payment->update(['status' => 'failed']);
//            return $payment;
//        }
//    }
//}


namespace App\Services;

use App\Models\Payment;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class PaymentService
{
    protected string $merchantId;
    protected bool $isSandbox;
    protected string $requestUrl;
    protected string $verifyUrl;
    protected string $redirectUrl;

    public function __construct()
    {
        $this->merchantId = config('services.zarinpal.merchant_id');
        $this->isSandbox = config('services.zarinpal.sandbox', false);

        if ($this->isSandbox) {
            $this->requestUrl = 'https://sandbox.zarinpal.com/pg/v4/payment/request.json';
            $this->verifyUrl = 'https://sandbox.zarinpal.com/pg/v4/payment/verify.json';
            $this->redirectUrl = 'https://sandbox.zarinpal.com/pg/StartPay/';
        } else {
            $this->requestUrl = 'https://api.zarinpal.com/pg/v4/payment/request.json';
            $this->verifyUrl = 'https://api.zarinpal.com/pg/v4/payment/verify.json';
            $this->redirectUrl = 'https://www.zarinpal.com/pg/StartPay/';
        }
    }

    /**
     * Request a payment URL from Zarinpal.
     *
     * @param Payment $payment
     * @return string|null The redirect URL or null on failure.
     */
    public function requestPayment(Payment $payment): ?string
    {
        if (empty($this->merchantId)) {
            Log::error('Zarinpal Merchant ID is not configured.');
            return null;
        }

        try {
            // [FIX] Added withoutVerifying() to bypass local SSL issues.
            $response = Http::withoutVerifying()->post($this->requestUrl, [
                'merchant_id' => $this->merchantId,
                'amount' => $payment->amount * 10, // Converting Toman to Rial
                'callback_url' => route('payment.callback', ['uuid' => $payment->uuid]),
                'description' => 'پرداخت برای درخواست ' . $payment->payable->purchaseRequest->request_code,
                'metadata' => [
                    'email' => $payment->user->email,
                ],
            ]);

            if ($response->successful() && $response->json('data.code') === 100) {
                $authority = $response->json('data.authority');
                $payment->update(['transaction_id' => $authority]);
                return $this->redirectUrl . $authority;
            } else {
                Log::error('Zarinpal request failed.', ['response' => $response->body()]);
                return null;
            }
        } catch (\Exception $e) {
            Log::error('Exception during Zarinpal payment request.', ['message' => $e->getMessage()]);
            return null;
        }
    }

    /**
     * Verify the payment after the user returns from Zarinpal.
     *
     * @param array $callbackData Data from the callback URL (e.g., request()->all())
     * @return Payment|null The updated payment record or null on failure.
     */
    public function verifyPayment(array $callbackData): ?Payment
    {
        $authority = $callbackData['Authority'] ?? null;
        $status = $callbackData['Status'] ?? null;

        if (!$authority || !$status) {
            return null;
        }

        $payment = Payment::where('transaction_id', $authority)->firstOrFail();

        if ($status !== 'OK') {
            $payment->update(['status' => 'failed']);
            return $payment;
        }

        try {
            // [FIX] Added withoutVerifying() to bypass local SSL issues.
            $response = Http::withoutVerifying()->post($this->verifyUrl, [
                'merchant_id' => $this->merchantId,
                'amount' => $payment->amount * 10, // Converting Toman to Rial
                'authority' => $authority,
            ]);

            if ($response->successful() && $response->json('data.code') === 100) {
                $payment->update([
                    'status' => 'successful',
                    'paid_at' => now(),
                    'meta' => ['ref_id' => $response->json('data.ref_id')],
                ]);
                return $payment;
            } else {
                $payment->update(['status' => 'failed', 'meta' => $response->json('errors')]);
                Log::error('Zarinpal verification failed.', ['response' => $response->body()]);
                return $payment;
            }
        } catch (\Exception $e) {
            Log::error('Exception during Zarinpal payment verification.', ['message' => $e->getMessage()]);
            $payment->update(['status' => 'failed']);
            return $payment;
        }
    }
}
